macroScript FunctionExplorer
category:"fooTOOLS"
buttontext:"FunctionExplorer"
tooltip:"FunctionExplorer - Show the graph of f(x) from [-1,-1] to [1,1]"
icon:#("fooTOOLS-Icons",9)
(

------------------------------------------------------------------------------------------
-- Contents:
--		FunctionExplorer - Show the graph of f(x) from [-1,-1] to [1,1]
--
-- Requires:
--		jbFunctions.ms
------------------------------------------------------------------------------------------

if (
	if (jbFunctionsCurrentVersion == undefined OR (jbFunctionsCurrentVersion() < 11)) then (
		local str = "This script requires jbFunctions to run properly.\n\nYou can get the latest version at http://www.footools.com/.\n\nWould you like to connect there now?"
		if (QueryBox str title:"Error") then ( try (ShellLaunch "http://www.footools.com/" "") catch () )
		FALSE
	) else (
		jbFunctionsVersionCheck #( #("jbFunctions",14) )
	)
) then (

	local thisTool = BFDtool	toolName:"FunctionExplorer"		\
								author:"John Burnett"			\
								createDate:[2000,01,07]			\
								modifyDate:[2001,05,21]		\
								version:3						\
								defFloaterSize:[400,460]

	local presetExpr = #(
				"sin ( ($x*360) * (1+$a) + ($c*360) ) * ($b+1)",
				"sin ( ($x*360) * (1+$a) + ($c*360) ) * ($b+1) * (1-(abs (sin ($x*360))))",
				"bias $a $x",
				"if ($a<0.5) then ($a*$x*2) else (bias $a $x)",
				"pow 0.08 $p",
				"pow ($x+1) $p")
	local canvasSize = [150,150]

	local canvas = bitmap canvasSize.x canvasSize.y
	local EPSILON = 0.00001
	local statusOutputs = #()
	local INF = 999999.0

	rollout DLGaboutRollout "About" (
		button DLGhelp "Help"
		label DLGAbout01 "" offset:[0,5]
		label DLGAbout02 ""
		label DLGAbout03 ""

		on DLGhelp pressed do (
			local helpStr = "Function Explorer graphs user defined equations.

The equation can be any valid Maxscript expression.
There are four predefined variables that are available
for use:

$x: Represents the x position on the graph while the
    expression is being evaluated.
$a, $b, $c: Represents the 3 spinners available for use
$p: Represents the last computed y value

There is a small dropdown next to the function field, that
holds predefined functions.  This list can be modified by
editing the \"presetExpr\" array that is defined at the top
of the script itself.

On the left side of the graph there is an array of numbers
showing exactly what the function evaluates to at specific
points, along with a spinner that lets you directly sample
the function at a specific point (this spinner is not bound
to -1.0 < x < 1.0)"
			messageBox helpStr title:"Function Explorer Help"
		)

		on DLGaboutRollout open do (
			DLGabout01.text = thisTool.toolName
			DLGabout02.text = thisTool.author
			DLGabout03.text =	(thisTool.modifyDate.x as integer) as string + "." +
								(thisTool.modifyDate.y as integer) as string + "." +
								(thisTool.modifyDate.z as integer) as string
		)

		on DLGaboutRollout close do ( thisTool.closeTool() )
	)

	rollout DLGmainRollout "Main Rollout" (
		fn toUpper instring = (
			local upper="ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			local lower="abcdefghijklmnopqrstuvwxyz"

			for i in 1 to instring.count do (
				j = findString lower instring[i]
				if (j != undefined) do instring[i] = upper[j]
			)
		)

		fn replaceVar str oldChar newChar = (
			local idx
			while ((idx = findString str oldChar) != undefined) do (
				str = replace str idx 2 newChar
			)
			return str
		)

		fn CreateEmptyGraph xsize ysize = (
			local R = DLGmainRollout
			local graphBackCol = color 0 32 0
			local graphOriginCol = color 0 128 0
			local graphGridCol = color 0 64 0

			local bmp = bitmap xsize ysize color:graphBackCol

			local halfX = (xsize / 2) as integer
			local x = halfX - (halfX * R.DLGxpos.value * R.DLGxzoom.value)
			for y in 0 to (ysize-1) do (
				SetPixels bmp [x,y] #(graphOriginCol)
			)

			local horizLine = #(); horizLine[xsize] = undefined;
			for i in 1 to xsize do ( horizLine[i] = graphOriginCol )
			local halfY = (ysize / 2) as integer
			local y = halfY + (halfY * R.DLGypos.value * R.DLGyzoom.value)
			SetPixels bmp [0,y] horizLine

			return bmp
		)

		fn SampleGraph	exprStr startSample endSample numSamples scaleFactor		\
						samples sampleType =
		(
			try (
				toUpper exprStr
				exprStr = replaceVar exprStr "$A" (DLGmainRollout.DLGspinA.value as string)
				exprStr = replaceVar exprStr "$B" (DLGmainRollout.DLGspinB.value as string)
				exprStr = replaceVar exprStr "$C" (DLGmainRollout.DLGspinC.value as string)

				local d = (endSample - startSample) / (numSamples-1.0)

				local minVal, maxVal
				if (startSample < endSample) then (
					minVal = startSample
					maxVal = endSample
				) else (
					minVal = endSample
					maxVal = startSample
				)

				local x = startSample
				local py = 0.0
				for i in 1 to numSamples do (
					local tmpStr = replaceVar exprStr "$X" (x as string)
					tmpStr = ReplaceVar tmpStr "$P" (py as string)

					local y = try (execute tmpStr) catch (undefined)

					if ((classOf y != float) AND (classOf y != integer)) then (
						y = 0
						sampleType[i] = #nan
					) else if (y > INF) then (
						y = 0
						sampleType[i] = #inf
					) else if (abs y > INF) then (
						y = 0
						sampleType[i] = #ninf
					) else (
						sampleType[i] = #good
					)

					samples[i] = [x,y,0] * scaleFactor

					-- doing this to avoid round off errors (sometimes last sample would be
					-- dropped when using a "for x in start to end by d" loop
					x += d
					x = if (x < minVal) then minVal else if (x > maxVal) then maxVal else x
					py = y
				)

				TRUE
			) catch (
				FALSE
			)
		)

		fn GetSamples numSamples samples sampleType scaleFactor:1.0 =
		(
			local R = DLGmainRollout
			local startSample = R.DLGxpos.value - (1.0 / R.DLGxzoom.value)
			local endSample = R.DLGxpos.value + (1.0 / R.DLGxzoom.value)

			if NOT SampleGraph DLGmainRollout.DLGexpr.text	\
								startSample					\
								endSample					\
								numSamples					\
								scaleFactor					\
								samples sampleType then return FALSE

			local ny
			local yOffset = R.DLGypos.value
			local yScale = R.DLGyzoom.value
			for i in 1 to samples.count do (
				ny = samples[i].y
				samples[i].y = (samples[i].y - yOffset) * yScale
			)

			TRUE
		)

		fn PlotGraph =
		(
			local R = DLGmainRollout
			local bmp = copy canvas

			local nx, ny, y
			local py
			local maxX = (bmp.width - 1) as float
			local maxY = (bmp.height - 1) as float
			local halfX = bmp.width / 2.0
			local halfY = bmp.height / 2.0
			local exprStr = R.DLGexpr.text

			toUpper exprStr

			exprStr = replaceVar exprStr "$A" (R.DLGspinA.value as string)
			exprStr = replaceVar exprStr "$B" (R.DLGspinB.value as string)
			exprStr = replaceVar exprStr "$C" (R.DLGspinC.value as string)

			local samples = #(); samples[bmp.width] = undefined
			local sampleType = #(); sampleType[bmp.width] = undefined
			if NOT GetSamples bmp.width samples sampleType then return FALSE

			for x in 0 to maxX do
			(
				ny = samples[x+1].y
				case sampleType[x+1] of
				(
					#nan: SetPixels bmp [x,halfY] #(red)
					#inf: SetPixels bmp [x,0] #(red)
					#ninf: SetPixels bmp [x,maxY] #(red)
					default: (
						y = (ny+1) * halfY
						y = if (y > maxY) then (maxY) else if (y < 0) then (0) else (y)
						SetPixels bmp [x,(maxY-y)] #(white)
					)
				)
			)

			if R.DLGsampleGraph.checked then (
				samples = #(); samples[11] = undefined
				sampleType = #(); sampleType[11] = undefined
				if NOT GetSamples 11 samples sampleType then return FALSE

				for i in 1 to 11 do
				(
					if (sampleType[i] == #good) then (
						local nx = if ((abs samples[i].x) < EPSILON) then 0.0 else samples[i].x
						local ny = if ((abs samples[i].y) < EPSILON) then 0.0 else samples[i].y
						local nxpad = if nx >= 0.0 then " " else ""
						local nypad = if ny >= 0.0 then " " else ""
						str = 	"f(" + nxpad as string + nx as string + ") = " + nypad as string + ny as string
						statusOutputs[i].text = str
					)
				)

				nx = R.DLGsampleAt.value
				if NOT (SampleGraph DLGmainRollout.DLGexpr.text nx nx 1 1.0 samples sampleType) then return FALSE

				if (sampleType[1] == #good) then
				(
					local x = ((samples[1].x - R.DLGxpos.value) * R.DLGxzoom.value + 1) * halfX

					local y = maxY - ((samples[1].y - R.DLGypos.value) * R.DLGyzoom.value + 1) * halfY

					setPixels bmp [x,y] #(red)
					setPixels bmp [x+1,y] #(red)
					setPixels bmp [x-1,y] #(red)
					setPixels bmp [x,y+1] #(red)
					setPixels bmp [x,y-1] #(red)
					R.DLGsampleAtOutput.text = "Y = " + (samples[1].y as string)
				)
			)

			R.DLGgraph.bitmap = bmp
			gc()
		)

		fn UpdateUI =
		(
			local R = DLGmainRollout

			if NOT R.DLGsampleGraph.checked then ( for i in statusOutputs do i.text = "[-,-]" )
			for i in statusOutputs do i.enabled = R.DLGsampleGraph.checked
			R.DLGsampleAt.enabled =
				R.DLGsampleAtLabel.enabled =
				R.DLGsampleAtOutput.enabled = R.DLGsampleGraph.checked
		)

		group "Variables" (
			spinner DLGspinA "$A:" range:[-999999,999999,0] type:#float scale:0.01 width:70 across:3
			spinner DLGspinB "$B:" range:[-999999,999999,0] type:#float scale:0.01 width:70
			spinner DLGspinC "$C:" range:[-999999,999999,0] type:#float scale:0.01 width:70
		)
		group "Function" (
			dropdownlist DLGpresets "" width:15 items:presetExpr align:#left
			edittext DLGexpr "" text:presetExpr[1] width:325 offset:[15,-25]
		)
		checkbox DLGsampleGraph "Sample Graph" checked:FALSE
		label DLGoutput01 "" align:#left offset:[0,-5]
		label DLGoutput02 "" align:#left offset:[0,-5]
		label DLGoutput03 "" align:#left offset:[0,-5]
		label DLGoutput04 "" align:#left offset:[0,-5]
		label DLGoutput05 "" align:#left offset:[0,-5]
		label DLGoutput06 "" align:#left offset:[0,-5]
		label DLGoutput07 "" align:#left offset:[0,-5]
		label DLGoutput08 "" align:#left offset:[0,-5]
		label DLGoutput09 "" align:#left offset:[0,-5]
		label DLGoutput10 "" align:#left offset:[0,-5]
		label DLGoutput11 "" align:#left offset:[0,-5]
		label DLGsampleAtLabel "At X:" align:#left offset:[0,0]
		spinner DLGsampleAt "" range:[-999999,999999,0] type:#float scale:0.01 fieldWidth:40 align:#left offset:[30,-19]
		label DLGsampleAtOutput "-0.999999" align:#left offset:[90,-20]
		bitmap DLGgraph "" bitmap:canvas offset:[95,-175]
		spinner DLGxpos "Pos X:" range:[-999999,999999,0] type:#float scale:0.01 fieldWidth:40 align:#right offset:[-72,-5]
		spinner DLGypos "Y:" range:[-999999,999999,0] type:#float scale:0.01 fieldWidth:40 align:#right offset:[0,-21]
		spinner DLGxzoom "Zoom X:" range:[0.00001,999999,1.0] type:#float scale:0.01 fieldWidth:40 align:#right offset:[-72,-2]
		spinner DLGyzoom "Y:" range:[0.00001,999999,1.0] type:#float scale:0.01 fieldWidth:40 align:#right offset:[0,-21]

		group "Output Function" (
			spinner DLGnumSamples "Number Of Samples" range:[2,99999,101] type:#integer fieldWidth:60 across:2
			spinner DLGscaleFactor "Scale Factor" range:[0,100000,1] type:#float fieldWidth:60
			button DLGcreateSpline "Create Spline" width:100 across:2
			button DLGprintArray "Print Array" width:100
		)

		on DLGspinA changed val do ( plotGraph() )
		on DLGspinB changed val do ( plotGraph() )
		on DLGspinC changed val do ( plotGraph() )
		on DLGpresets selected idx do (
			DLGexpr.text = presetExpr[idx]
			plotGraph()
		)
		on DLGexpr entered str do ( plotGraph() )
		on DLGsampleAt changed val do ( plotGraph() )
		on DLGsampleGraph changed state do (
			UpdateUI()
			plotGraph()
		)
		on DLGxpos changed val do ( canvas = CreateEmptyGraph canvasSize.x canvasSize.y; PlotGraph() )
		on DLGypos changed val do ( canvas = CreateEmptyGraph canvasSize.x canvasSize.y; PlotGraph() )
		on DLGxzoom changed val do ( canvas = CreateEmptyGraph canvasSize.x canvasSize.y; PlotGraph() )
		on DLGyzoom changed val do ( canvas = CreateEmptyGraph canvasSize.x canvasSize.y; PlotGraph() )

		on DLGcreateSpline pressed do (
			local samples = #(); samples[DLGnumSamples.value] = undefined
			local sampleType = #(); sampleType[DLGnumSamples.value] = undefined

			if NOT (GetSamples DLGnumSamples.value samples sampleType scaleFactor:DLGscaleFactor.value) then return()

			for i in 1 to samples.count do samples[i].y = if sampleType[i] == #good then samples[i].y else 0

			local obj = CreateSplineFromArray samples

			if (obj != undefined) then select obj
		)

		on DLGprintArray pressed do (
			local samples = #(); samples[DLGnumSamples.value] = undefined
			local sampleType = #(); sampleType[DLGnumSamples.value] = undefined

			if NOT (GetSamples DLGnumSamples.value samples sampleType scaleFactor:DLGscaleFactor.value) then return()

			format "#("
			for i in 1 to (samples.count-1) do (
				format "[%, %]," samples[i].x samples[i].y
				if (mod i 10 == 0) then format "\t\\\n"
			)
			format "[%, %])\n" samples[samples.count].x samples[samples.count].y
		)

		on DLGmainRollout open do (
			canvas = CreateEmptyGraph canvasSize.x canvasSize.y
			statusOutputs = #(	DLGoutput01, DLGoutput02, DLGoutput03, DLGoutput04,
								DLGoutput05, DLGoutput06, DLGoutput07, DLGoutput08,
								DLGoutput09, DLGoutput10, DLGoutput11 )
			UpdateUI()
			plotGraph()
		)
	)

	thisTool.addRoll #(DLGaboutRollout, DLGmainRollout) rolledUp:#(true,false)

	thisTool.openTool thisTool
)
)
