macroscript ReallyAdvancedQuadOptions
category:"Customize User Interface"
buttonText:"Really Advanced Quad Options"
tooltip:"Really Advanced Quad Options"
(
	/**************************************************yay*for*multi*line*comments*
	Description:	UI for QuadMenuSettings
	Author:			John Burnett				john@johnburnett.com
	*******************************************************************************/

	-- Predefined list of font names, add/remove to taste
	-- The list below was taken from a fresh install of Win2000, so they should (hopefully)
	-- be available to most users.  Selecting one not installed on the current system just
	-- makes the quadmenu use a default font.
	local fontList = #(
		"",
		"Arial","    Arial Bold","    Arial Bold Italic","    Arial Italic",
		"Arial Black",
		"Comic Sans MS","    Comic Sans MS Bold",
		"Courier New","    Courier New Bold","    Courier New Bold Italic","    Courier New Italic",
		"Georgia","    Georgia Bold","    Georgia Bold Italic","    Georgia Italic",
		"Impact",
		"Lucida Console",
		"Lucida Sans Unicode",
		"Microsoft Sans Serif Regular",
		"Palatino Linotype","    Palatino Linotype Bold","    Palatino Linotype Bold Italic","    Palatino Linotype Italic",
		"Symbol",
		"Tahoma","    Tahoma Bold",
		"Times New Roman","    Times New Roman Bold","    Times New Roman Bold Italic","    Times New Roman Italic",
		"Trebuchet MS","    Trebuchet MS Bold","    Trebuchet MS Bold Italic","    Trebuchet MS Italic",
		"Verdana","    Verdana Bold","    Verdana Bold Italic","    Verdana Italic",
		"Webdings",
		"WingDings"
	)
	-- This list is a bit more varied, but I don't think all of these are standard
	/*local fontList = #(
		"",
		"Arial","    Arial Bold","    Arial Bold Italic","    Arial Italic",
		"Arial Black","    Arial Black Italic",
		"Arial Narrow","    Arial Narrow Bold","    Arial Narrow Bold Italic","    Arial Narrow Italic",
		"Book Antiqua","    Book Antiqua Bold","    Book Antiqua Bold Italic","    Book Antiqua Italic",
		"Bookman Old Style","    Bookman Old Style Bold","    Bookman Old Style Bold Italic","    Bookman Old Style Italic",
		"Century Gothic","    Century Gothic Bold","    Century Gothic Bold Italic","    Century Gothic Italic",
		"Comic Sans MS","    Comic Sans MS Bold",
		"Courier New","    Courier New Bold","    Courier New Bold Italic","    Courier New Italic",
		"Garamond","    Garamond Bold","    Garamond Italic",
		"Georgia","    Georgia Bold","    Georgia Bold Italic","    Georgia Italic",
		"Haettenschweiler",
		"Impact",
		"Lucida Console",
		"Lucida Sans Unicode",
		"Microsoft Sans Serif Regular",
		"Monotype Sorts",
		"MS Gothic",
		"MS Mincho",
		"Palatino Linotype","    Palatino Linotype Bold","    Palatino Linotype Bold Italic","    Palatino Linotype Italic",
		"Symbol",
		"Tahoma","    Tahoma Bold",
		"Times New Roman","    Times New Roman Bold","    Times New Roman Bold Italic","    Times New Roman Italic",
		"Trebuchet MS","    Trebuchet MS Bold","    Trebuchet MS Bold Italic","    Trebuchet MS Italic",
		"Verdana","    Verdana Bold","    Verdana Bold Italic","    Verdana Italic",
		"Webdings",
		"WingDings"
	)*/

--	local STARTUP_FILENAME = (GetDir #startupScripts) + "\\QuadMenuSettings_Startup.ms"
	local STARTUP_FILENAME = (GetDir #maxRoot) + "Stdplugs\\Stdscripts\\QuadOptions_Startup.ms"

	-- Indicies of rollout controls for certain arrays
	-- MAKE SURE THESE ARE UP TO DATE IF NEW CONTROLS ARE ADDED
	local COLORSWATCHES_START = 39 -- first control of the color swatches
	local COLORSWATCHES_COUNT = 10 -- how many color swatches there are PER QUAD
	local COLORLOCKS_START = 29
	local COLORLOCKS_END = 38
	local QUADCOMBO_START = 117
	local QUADCOMBO_END = 120

	local QMS = QuadMenuSettings -- shortcut to typing QuadMenuSettings repeatedly

	local needSave = false -- track whether save on exit prompt is needed

	local cursorPosBitmap = bitmap 53 53 -- bitmap for showing cursor position
	local colorIcons = #() -- pretty icons for color swatch labels

	-- The following just stuffs dialog items and their associated functions into
	-- arrays so they can be dealt with en-mass by iterating over the arrays
	local colorSwatch -- array of arrays of color swatches: colorSwatch[quadNum][colorSwatchNum]
	local colorLock -- array of lock buttons: colorLock[colorSwatchNum]
	-- Array of functions used to get above color swatch colors
	local GetQuadColor = #(
		QMS.GetDisabledHighlightColor,QMS.GetDisabledShadowColor,
		QMS.GetBorderColor,QMS.GetItemTextColor,QMS.GetItemBackgroundColor,
		QMS.GetHighlightedItemTextColor,QMS.GetHighlightedItemBackgroundColor,QMS.GetLastExecutedItemTextColor,
		QMS.GetTitleBarTextColor,QMS.GetTitleBarBackgroundColor
	)
	-- Array of functions used to set above color swatch colors
	local SetQuadColor = #(
		QMS.SetDisabledHighlightColor,QMS.SetDisabledShadowColor,
		QMS.SetBorderColor,QMS.SetItemTextColor,QMS.SetItemBackgroundColor,
		QMS.SetHighlightedItemTextColor,QMS.SetHighlightedItemBackgroundColor,QMS.SetLastExecutedItemTextColor,
		QMS.SetTitleBarTextColor,QMS.SetTitleBarBackgroundColor
	)

	-- Array of registered quad menu names
	local quadKeyCombos = #(
		#nonePressed, #shiftPressed, #altPressed, #controlPressed,
		#shiftAndAltPressed, #shiftAndControlPressed, #controlAndAltPressed,
		#shiftAndAltAndControlPressed
	)
	local quadKeyComboStrings = #(
		"Default", "Shift", "Alt", "Control",
		"Shift+Alt", "Shift+Control", "Control+Alt",
		"Shift+Alt+Control"
	)

	fn LoadStartupFile = (
		local res = try ( fileIn STARTUP_FILENAME; true ) catch ( false )

		if res then (
			local str = "Startup Script Loaded:\n\n" + STARTUP_FILENAME
			Messagebox str title:"Quad Menu Settings"
		) else (
			local str = "Failed To Load Startup Script:\n\n" + STARTUP_FILENAME
			MessageBox str title:"Warning"
		)

		needSave = NOT res

		res
	)

	fn SaveStartupFile = (
		local f

		local res = try (
			f = Createfile STARTUP_FILENAME

			-- colors
			local fnName
			for quadIdx in 1 to 4 do (
				for colIdx in 1 to SetQuadColor.count do (
					fnName = SetQuadColor[colIdx] as string
					fnName = SubString fnName 1 (fnName.count-2)
					format "QuadMenuSettings.% % %\n" fnName quadIdx (GetQuadColor[colIdx] quadIdx) to:f
				)
			)

			-- interaction
			format "QuadMenuSettings.SetUseLastExecutedItem %\n" (QMS.GetUseLastExecutedItem()) to:f

			-- animation
			format "QuadMenuSettings.SetDisplayMethod %\n" (QMS.GetDisplayMethod()) to:f
			format "QuadMenuSettings.SetAnimatedStepTime %\n" (QMS.GetAnimatedStepTime()) to:f
			format "QuadMenuSettings.SetAnimatedSteps %\n" (QMS.GetAnimatedSteps()) to:f

			-- display
			format "QuadMenuSettings.SetSubMenuPauseTime %\n" (QMS.GetSubMenuPauseTime()) to:f
			format "QuadMenuSettings.SetOpacity %\n" (QMS.GetOpacity()) to:f
			format "QuadMenuSettings.SetRemoveRedundantSeparators %\n" (QMS.GetRemoveRedundantSeparators()) to:f

			-- position
			format "QuadMenuSettings.SetInitialCursorLocInBox_0to1 % %\n" (QMS.GetInitialCursorLocXInBox_0to1()) (QMS.GetInitialCursorLocYInBox_0to1()) to:f
			format "QuadMenuSettings.SetFirstQuadDisplayed %\n" (QMS.GetFirstQuadDisplayed()) to:f
			format "QuadMenuSettings.SetMirrorQuad %\n" (QMS.GetMirrorQuad()) to:f
			format "QuadMenuSettings.SetRepositionWhenClipped %\n" (QMS.GetRepositionWhenClipped()) to:f
			format "QuadMenuSettings.SetMoveCursorOnReposition %\n" (QMS.GetMoveCursorOnReposition()) to:f
			format "QuadMenuSettings.SetReturnCursorAfterReposition %\n" (QMS.GetReturnCursorAfterReposition()) to:f

			-- size
			format "QuadMenuSettings.SetBorderSize %\n" (QMS.GetBorderSize()) to:f
			format "QuadMenuSettings.SetUseUniformQuadWidth %\n" (QMS.GetUseUniformQuadWidth()) to:f
			format "QuadMenuSettings.SetUseUniformItemHeight %\n" (QMS.GetUseUniformItemHeight()) to:f
			format "QuadMenuSettings.SetHorizontalMarginInPoints %\n" (QMS.GetHorizontalMarginInPoints()) to:f
			format "QuadMenuSettings.SetVerticalMarginInPoints %\n" (QMS.GetVerticalMarginInPoints()) to:f

			-- fonts
			format "QuadMenuSettings.SetItemFontFace \"%\"\n" (QMS.GetItemFontFace()) to:f
			format "QuadMenuSettings.SetItemFontSize %\n" (QMS.GetItemFontSize()) to:f
			format "QuadMenuSettings.SetTitleFontFace \"%\"\n" (QMS.GetTitleFontFace()) to:f
			format "QuadMenuSettings.SetTitleFontSize %\n" (QMS.GetTitleFontSize()) to:f

			true
		) catch ( false )

		try (close f) catch ()

		-- explicitly save QuadMenu -> Key combo assignments
		-- this shouldn't really be in this function, but it's convenient...
		local menuRes = MenuMan.SaveMenuFile ( FilenameFromPath (MenuMan.GetMenuFile()) )

		if res AND menuRes then (
			local str = "Startup Script Saved:\n\n" + STARTUP_FILENAME
			Messagebox str title:"Quad Menu Settings"
		) else (
			local str = ""
			if NOT res then str += "FAILED TO SAVE STARTUP SCRIPT:\n" + STARTUP_FILENAME + "\n\n"
			if NOT menuRes then str += "FAILED TO SAVE MENU FILE:\n" + (MenuMan.GetMenuFile()) + "\n\n"
			str += "Check that the appropriate file(s) above are not read only."
			MessageBox str title:"WARNING"
		)

		needSave = NOT res

		res
	)

	-- Strip out extra spaces in a string
	fn StripSpaces aString = (
		local strAr = filterString aString " "
		local newStr = ""
		for str in strAr do newStr += (str + " ")
		subString newStr 1 (newStr.count-1)
	)

	-- return num clamped between minVal and maxVal
	fn Clamp num minVal maxVal = (
		if (num > maxVal) then maxVal else if (num < minVal) then minVal else num
	)
	fn ClampPnt2 num minVal maxVal = (
		[(Clamp num.x minVal maxVal), (Clamp num.y minVal maxVal)]
	)

	-- invert a color
	fn InvertColor col = (
		color (abs (col.r-255)) (abs (col.g-255)) (abs (col.b-255))
	)

	-- draw a representative quad icon with "quadNum" highlighted
	-- if quadNum == 0, no quads are highlighted
	-- if quadNum == 5, all quads are highlighted
	fn DrawQuadIcon bmp quadNum = (
		local halfW = bmp.width/2
		local halfH = bmp.height/2

		-- fill in background color
		local fillCol = if (quadNum > 4) then QMS.GetHighlightedItemBackgroundColor(quadNum) else QMS.GetTitleBarBackgroundColor(quadNum)
		local fillRow = for x in 1 to bmp.width collect fillCol
		for y in 0 to (bmp.height-1) do setPixels bmp [0,y] fillRow

		-- highlight quad
		if (quadNum >= 1) AND (quadNum <= 4) then (
			local startPos
			local endPos
			case quadNum of (
				1: ( startPos = [halfW,halfH];	endPos = [bmp.width,bmp.height] )
				2: ( startPos = [halfW,0];		endPos = [bmp.width,halfH] )
				3: ( startPos = [0,0];			endPos = [halfW,halfH] )
				4: ( startPos = [0,halfH];		endPos = [halfW,bmp.height] )
			)

			local quadRow = for x in startPos.x to (endPos.x-1) collect QMS.GetHighlightedItemBackgroundColor(quadNum)
			for y in startPos.y to endPos.y do setPixels bmp [startPos.x,y] quadRow
		)

		-- draw grid
		local lineRow = for x in 1 to bmp.width collect QMS.GetBorderColor(quadNum)
		setPixels bmp [0,halfH] lineRow
		lineRow = #(QMS.GetBorderColor(quadNum))
		for y in 0 to (bmp.height-1) do setPixels bmp [halfW,y] lineRow

		bmp
	)

	-- draw a crosshair on bitmap using inverted colors
	-- at nCrossPos (a point2 normalized to 0.0 to 1.0)
	fn DrawCrosshair bmp nCrossPos crossSize:10 = (

		crossPos = [(bmp.width-1),(bmp.height-1)] * nCrossPos

		local halfCross = crossSize/2
		hLine = [crossPos.x-halfCross,crossPos.x+halfCross]
		vLine = [crossPos.y-halfCross,crossPos.y+halfCross]
		hLine = ClampPnt2 hLine 0 (bmp.width-1)
		vLine = ClampPnt2 vLine 0 (bmp.height-1)

		local crossColor
		-- draw top horizontal line
		if (vLine.x >= 0) AND (vLine.x < bmp.height) then (
			for x in hLine.x to hLine.y do (
				crossColor = InvertColor (getPixels bmp [x,vLine.x] 1)[1]
				setPixels bmp [x,vLine.x] #(crossColor)
			)
		)
		-- draw middle horizontal line
		if (crossPos.y >= 0) AND (crossPos.y < bmp.height) then (
			for x in hLine.x to hLine.y do (
				crossColor = InvertColor (getPixels bmp [x,crossPos.y] 1)[1]
				setPixels bmp [x,crossPos.y] #(crossColor)
			)
		)
		-- draw lower horizontal line
		if (vLine.y >= 0) AND (vLine.y < bmp.height) then (
			for x in hLine.x to hLine.y do (
				crossColor = InvertColor (getPixels bmp [x,vLine.y] 1)[1]
				setPixels bmp [x,vLine.y] #(crossColor)
			)
		)
		-- draw left vertical line
		if (hLine.x >= 0) AND (hLine.x < bmp.width) then (
			for y in vLine.x to vLine.y do (
				crossColor = InvertColor (getPixels bmp [hLine.x,y] 1)[1]
				setPixels bmp [hLine.x,y] #(crossColor)
			)
		)
		-- draw middle vertical line
		if (crossPos.x >= 0) AND (crossPos.x < bmp.width) then (
			for y in vLine.x to vLine.y do (
				crossColor = InvertColor (getPixels bmp [crossPos.x,y] 1)[1]
				setPixels bmp [crossPos.x,y] #(crossColor)
			)
		)
		-- draw right vertical line
		if (hLine.y >= 0) AND (hLine.y < bmp.width) then (
			for y in vLine.x to vLine.y do (
				crossColor = InvertColor (getPixels bmp [hLine.y,y] 1)[1]
				setPixels bmp [hLine.y,y] #(crossColor)
			)
		)

		bmp
	)

	-- draw color label icons
	fn DrawColorLabelIcons = (
		for i in 1 to 4 do (
			colorIcons[i] = bitmap 24 24
			DrawQuadIcon colorIcons[i] i
		)
		OK
	)

	DrawColorLabelIcons()

	fn ImportUIColors = (
		for i in 1 to 4 do (
			QMS.SetDisabledHighlightColor i ((colorMan.getColor #hilight)*255)
			QMS.SetDisabledShadowColor i ((colorMan.getColor #shadow)*255)
			QMS.SetBorderColor i ((colorMan.getColor #text)*255)
			QMS.SetItemTextColor i ((colorMan.getColor #text)*255)
			QMS.SetItemBackgroundColor i ((colorMan.getColor #background)*255)
			QMS.SetHighlightedItemTextColor i ((colorMan.getColor #hilightText)*255)
			QMS.SetHighlightedItemBackgroundColor i ((colorMan.getColor #activeCommand)*255)
			local col = ((colorMan.getColor #activeCommand)*255) as color
			QMS.SetLastExecutedItemTextColor i (InvertColor col)
			QMS.SetTitleBarTextColor i ((colorMan.getColor #rollupTitleText)*255)
			QMS.SetTitleBarBackgroundColor i ((colorMan.getColor #rollupTitleFace)*255)
		)
		OK
	)

	global QuadMenuSettingsRollout
	rollout QuadMenuSettingsRollout "Really Advanced Quad Options" width:756 height:390 (
		-- shortcut to typing rollout variable name repeatedly
		local R = QuadMenuSettingsRollout

		-- font list converted to names with no spaces, for easier searching
		local fontListNames

		-- takes a point2 in the range [-1,-1] to [1,1] and sets the
		-- InitialCursorLoclInBox and FirstQuadDisplayed accordingly
		fn SetCursorPos pos = (
			case of (
--TODO: add one to all SetFirstQuadDisplayed calls below if/when it becomes 1 based
				((pos.x >= -0.001) AND (pos.y <= 0.001)): (
					QMS.SetFirstQuadDisplayed 0
					QMS.SetInitialCursorLocInBox_0to1 pos.x -pos.y
				)
				((pos.x >= 0.0) AND (pos.y > 0.0)): (
					QMS.SetFirstQuadDisplayed 1
					QMS.SetInitialCursorLocInBox_0to1 pos.x pos.y
				)
				((pos.x < 0.0) AND (pos.y > 0.0)): (
					QMS.SetFirstQuadDisplayed 2
					QMS.SetInitialCursorLocInBox_0to1 -pos.x pos.y
				)
				((pos.x < 0.0) AND (pos.y <= 0.0)): (
					QMS.SetFirstQuadDisplayed 3
					QMS.SetInitialCursorLocInBox_0to1 -pos.x -pos.y
				)
			)
		)

		-- gets a point2 in the range [-1,-1] to [1,1] from the current
		-- InitialCursorLoclInBox and FirstQuadDisplayed settings
		fn GetCrosshairPos = (
			local pos = [0,0]

-- TODO: remove the +1 if/when GetFirstQuadDisplayed becomes 1 based
			case (QMS.GetFirstQuadDisplayed() + 1) of (
				1: (
					pos.x = QMS.GetInitialCursorLocXInBox_0to1()
					pos.y = -QMS.GetInitialCursorLocYInBox_0to1()
				)
				2: (
					pos.x = QMS.GetInitialCursorLocXInBox_0to1()
					pos.y = QMS.GetInitialCursorLocYInBox_0to1()
				)
				3: (
					pos.x = -(QMS.GetInitialCursorLocXInBox_0to1())
					pos.y = QMS.GetInitialCursorLocYInBox_0to1()
				)
				4: (
					pos.x = -(QMS.GetInitialCursorLocXInBox_0to1())
					pos.y = -(QMS.GetInitialCursorLocYInBox_0to1())
				)
			)
			-- potentially fix QuadMenuSettings so that FirstQuad
			-- and InitialCursorLoc don't conflict
			-- (which actually seems harmless enough anyway)
			SetCursorPos pos

			pos
		)

		-- draw cursor position bitmap, crosshair, and force UI refresh
		fn UpdateCursorPosBitmap = (
-- TODO: remove the +1 if/when GetFirstQuadDisplayed() switches to base 1
			DrawQuadIcon cursorPosBitmap (QMS.GetFirstQuadDisplayed()+1)

			-- switch cursorPos from -1 to 1 UI range to 0 to 1 for DrawCrosshair
			local cursorPos = [R.DLGinitialCursorPosX.value,R.DLGinitialCursorPosY.value]
			cursorPos = (cursorPos + 1.0) / 2.0
			cursorPos.y = 1.0-cursorPos.y
			DrawCrosshair cursorPosBitmap cursorPos

			-- force UI bitmap to refresh
			R.DLGcursorPosBitmap.bitmap = cursorPosBitmap
		)

		-- Set lock states on first open
		-- This happens before the UI is updated, so don't rely on it
		fn LoadUILocks = (
			-- set font lock if both fonts are the same
			R.DLGuseItemFont.checked = (QMS.GetItemFontFace() == QMS.GetTitleFontFace())
			-- loop through all quad color functions, compare colors of each quad, and
			-- set the color lock accordingly
			for i in 1 to colorLock.count do (
				colorLock[i].checked =	(GetQuadColor[i] 1 == GetQuadColor[i] 2) AND
										(GetQuadColor[i] 1 == GetQuadColor[i] 3) AND
										(GetQuadColor[i] 1 == GetQuadColor[i] 4)
			)
		)

		-- redraw all bitmaps and force UI refresh
		fn UpdateBitmaps = (
			UpdateCursorPosBitmap()
			DrawColorLabelIcons()
			R.DLGcolorsIcon01.bitmap = colorIcons[1]
			R.DLGcolorsIcon02.bitmap = colorIcons[2]
			R.DLGcolorsIcon03.bitmap = colorIcons[3]
			R.DLGcolorsIcon04.bitmap = colorIcons[4]
			OK
		)

		-- set colorswatch and quad color, while respecting color lock states
		fn UpdateQuadColor quadNum DLG = (
			-- find which item changed
			itemIdx = findItem colorSwatch[quadNum] DLG
			if colorLock[itemIdx].checked then (
				-- if it's locked, update all the others
				for i in 1 to 4 do (
					colorSwatch[i][itemIdx].color = DLG.color
					SetQuadColor[itemIdx] i DLG.color
				)
			) else (
				-- otherwise just update the one color
				SetQuadColor[itemIdx] quadNum DLG.color
			)
		)

		-- Get array of names of all registered quadmenus
		fn GetQuadMenuNames = (
			for i in 1 to menuMan.NumQuadMenus() collect (
				menuMan.GetQuadMenuName (menuMan.GetQuadMenu i)
			)
		)

		-- Load values into UI
		fn UpdateUI = (
			-- settings
			R.DLGloadStartup.enabled = (GetFiles STARTUP_FILENAME).count != 0

			-- color
			for quadIdx in 1 to 4 do (
				for swatchIdx in 1 to colorSwatch[quadIdx].count do (
					colorSwatch[quadIdx][swatchIdx].color = GetQuadColor[swatchIdx] quadIdx
				)
			)

			-- interaction
			R.DLGuseLastItem.checked = QMS.GetUseLastExecutedItem()

			-- animation
			R.DLGanimationMethod.selection = QMS.GetDisplayMethod() + 1
			R.DLGanimationStepTime.enabled =
				R.DLGanimationSteps.enabled = (R.DLGanimationMethod.selection != 1)
			R.DLGanimationStepTime.value = QMS.GetAnimatedStepTime()
			R.DLGanimationSteps.value = QMS.GetAnimatedSteps()

			-- display
			R.DLGsubMenuTime.value = QMS.GetSubMenuPauseTime()
			R.DLGopacity.value = QMS.GetOpacity() * 100
			R.DLGremoveRedundantSeparators.checked = QMS.GetRemoveRedundantSeparators()

			-- position
			R.DLGcursorPosBitmap.bitmap = R.DLGcursorPosBitmap.bitmap
			R.DLGinitialCursorPosX.value = (GetCrosshairPos()).x
			R.DLGinitialCursorPosY.value = (GetCrosshairPos()).y
			R.DLGmirrorQuad.checked = QMS.GetMirrorQuad()
			R.DLGrepositionWhenClipped.checked = QMS.GetRepositionWhenClipped()
			R.DLGmoveCursorOnReposition.checked = QMS.GetMoveCursorOnReposition()
			R.DLGreturnCursorAfterReposition.checked = QMS.GetReturnCursorAfterReposition()

			-- size
			R.DLGborderSize.value = QMS.GetBorderSize()
			R.DLGuniformWidth.checked = QMS.GetUseUniformQuadWidth()
			R.DLGuniformHeight.checked = QMS.GetUseUniformItemHeight()
			R.DLGhorizMargin.value = QMS.GetHorizontalMarginInPoints()
			R.DLGvertMargin.value = QMS.GetVerticalMarginInPoints()

			-- fonts
			local idx
			R.DLGitemFont.text = QMS.GetItemFontFace()
			idx = FindItem fontListNames (QMS.GetItemFontFace() as name)
			R.DLGitemFontList.selection =  if (idx != 0) then idx else 1
			R.DLGitemFontSize.value = QMS.GetItemFontSize()

			R.DLGtitleFont.text = QMS.GetTitleFontFace()
			idx = FindItem fontListNames (QMS.GetTitleFontFace() as name)
			R.DLGtitleFontList.selection = if (idx != 0) then idx else 1
			R.DLGtitleFontSize.value = QMS.GetTitleFontSize()

			R.DLGtitleFont.enabled =
				R.DLGtitleFontList.enabled = NOT R.DLGuseItemFont.checked

			-- quadMenu key combo
			local quadMenuNames = GetQuadMenuNames()
			R.DLGquadMenuNames.items = quadMenuNames
			if (0 == quadMenuNames.count) then (
				for i in QUADCOMBO_START to QUADCOMBO_END do R.controls[i].enabled = false
			) else (
				for i in QUADCOMBO_START to QUADCOMBO_END do R.controls[i].enabled = true

				-- find the quadmenu assigned to currently selected key combo
				local keyCombo = quadKeyCombos[R.DLGquadKeyCombo.selection]
				local quadMenu = menuMan.getViewportRightClickMenu keyCombo
				local quadName = menuMan.GetQuadMenuName quadMenu
				local idx = findItem quadMenuNames quadName

				if (idx != 0) then R.DLGquadMenuNames.selection = idx
			)

			UpdateBitmaps()
		)

		-- full reset and refresh of the UI
		fn ResetUI = (
			LoadUILocks()
			UpdateUI()
		)


--		groupBox DLGsettingsGroup "Settings" pos:[6,3] width:351 height:89
--			button DLGsaveStartup "Save Startup File" tooltip:"Save Settings To A Startup File" pos:[65,24] width:105 height:24
--			button DLGloadStartup "Load Startup File" tooltip:"Load Settings From Previously Saved Startup File" pos:[188,24] width:105 height:24
--			button DLGreset "Reset Settings To Max Defaults" tooltip:"Reset Settings To Max Defaults" pos:[89,54] width:180 height:24
		groupBox DLGsettingsGroup "Settings" pos:[6,3] width:139 height:89
			button DLGsaveStartup "Save Startup File" tooltip:"Save Settings To A Startup File" pos:[23,25] width:105 height:24
			button DLGloadStartup "Load Startup File" tooltip:"Load Settings From Previously Saved Startup File" pos:[23,56] width:105 height:24
		groupBox DLGaboutGroup "About" pos:[150,3] width:207 height:89
			label DLGaboutLabel01 "Really Advanced Quad Options" pos:[168,25] height:15
			label DLGaboutLabel02 "John Burnett" pos:[286,66] height:15
			label DLGaboutLabel03 "10.16.2000" pos:[292,46] height:15
		groupBox DLGcolorsGroup "Colors" pos:[6,97] width:351 height:288
			button DLGimportUIcolors "Import UI Colors" tooltip:"Assigns Quad Colors Based On General UI Colors" pos:[18,136] width:114 height:24
			label DLGcolorsLabel01 "Quad 1" pos:[144,117] width:36 height:13
			label DLGcolorsLabel02 "Quad 2" pos:[210,117] width:36 height:13
			label DLGcolorsLabel03 "Quad 3" pos:[260,117] width:36 height:13
			label DLGcolorsLabel04 "Quad 4" pos:[310,117] width:36 height:13
			bitmap DLGcolorsIcon01 "" bitmap:colorIcons[1] pos:[151,136] width:24 height:24
			bitmap DLGcolorsIcon02 "" bitmap:colorIcons[2] pos:[217,136] width:24 height:24
			bitmap DLGcolorsIcon03 "" bitmap:colorIcons[3] pos:[267,136] width:24 height:24
			bitmap DLGcolorsIcon04 "" bitmap:colorIcons[4] pos:[317,136] width:24 height:24
			label DLGdisabledHighlightLabel "Disabled Highlight:" pos:[44,173] width:89 height:13
			label DLGdisabledShadowLabel "Disabled Shadow:" pos:[46,193] width:87 height:13
			label DLGborderLabel "Border:" pos:[98,213] width:35 height:13
			label DLGtextLabel "Text:" pos:[108,233] width:25 height:13
			label DLGbackgroundLabel "Background:" pos:[71,253] width:62 height:13
			label DLGhighlightedTextLabel "Highlighted Text:" pos:[52,273] width:81 height:13
			label DLGhighlightedBackgroundLabel "Highlighted Background:" pos:[15,293] width:118 height:13
			label DLGlastUsedTextLabel "Last Used Text:" pos:[57,313] width:76 height:13
			label DLGtitleTextLabel "Title Text:" pos:[85,333] width:48 height:13
			label DLGtitleBackgroundLabel "Title Background:" pos:[48,353] width:85 height:13
			label DLGlockLabel "Lock" pos:[183,141] width:26 height:13
			checkbutton DLGdisabledHighlightLock "L" tooltip:"Lock Quad Colors" pos:[187,170] width:18 height:18
			checkbutton DLGdisabledShadowLock "L" tooltip:"Lock Quad Colors" pos:[187,190] width:18 height:18
			checkbutton DLGborderLock "L" tooltip:"Lock Quad Colors" pos:[187,210] width:18 height:18
			checkbutton DLGtextLock "L" tooltip:"Lock Quad Colors" pos:[187,230] width:18 height:18
			checkbutton DLGbackgroundLock "L" tooltip:"Lock Quad Colors" pos:[187,250] width:18 height:18
			checkbutton DLGhighlightedTextLock "L" tooltip:"Lock Quad Colors" pos:[187,270] width:18 height:18
			checkbutton DLGhighlightedBackgroundLock "L" tooltip:"Lock Quad Colors" pos:[187,290] width:18 height:18
			checkbutton DLGlastUsedTextLock "L" tooltip:"Lock Quad Colors" pos:[187,310] width:18 height:18
			checkbutton DLGtitleTextLock "L" tooltip:"Lock Quad Colors" pos:[187,330] width:18 height:18
			checkbutton DLGtitleBackgroundLock "L" tooltip:"Lock Quad Colors" pos:[187,350] width:18 height:18
			colorpicker DLGdisabledHighlight01 "" pos:[143,171] width:40 height:18
			colorpicker DLGdisabledShadow01 "" pos:[143,191] width:40 height:18
			colorpicker DLGborder01 "" pos:[143,211] width:40 height:18
			colorpicker DLGtext01 "" pos:[143,231] width:40 height:18
			colorpicker DLGbackground01 "" pos:[143,251] width:40 height:18
			colorpicker DLGhighlightedText01 "" pos:[143,271] width:40 height:18
			colorpicker DLGhighlightedBackground01 "" pos:[143,291] width:40 height:18
			colorpicker DLGlastUsedText01 "" pos:[143,311] width:40 height:18
			colorpicker DLGtitleText01 "" pos:[143,331] width:40 height:18
			colorpicker DLGtitleBackground01 "" pos:[143,351] width:40 height:18
			colorpicker DLGdisabledHighlight02 "" pos:[209,171] width:40 height:18
			colorpicker DLGdisabledShadow02 "" pos:[209,191] width:40 height:18
			colorpicker DLGborder02 "" pos:[209,211] width:40 height:18
			colorpicker DLGtext02 "" pos:[209,231] width:40 height:18
			colorpicker DLGbackground02 "" pos:[209,251] width:40 height:18
			colorpicker DLGhighlightedText02 "" pos:[209,271] width:40 height:18
			colorpicker DLGhighlightedBackground02 "" pos:[209,291] width:40 height:18
			colorpicker DLGlastUsedText02 "" pos:[209,311] width:40 height:18
			colorpicker DLGtitleText02 "" pos:[209,331] width:40 height:18
			colorpicker DLGtitleBackground02 "" pos:[209,351] width:40 height:18
			colorpicker DLGdisabledHighlight03 "" pos:[259,171] width:40 height:18
			colorpicker DLGdisabledShadow03 "" pos:[259,191] width:40 height:18
			colorpicker DLGborder03 "" pos:[259,211] width:40 height:18
			colorpicker DLGtext03 "" pos:[259,231] width:40 height:18
			colorpicker DLGbackground03 "" pos:[259,251] width:40 height:18
			colorpicker DLGhighlightedText03 "" pos:[259,271] width:40 height:18
			colorpicker DLGhighlightedBackground03 "" pos:[259,291] width:40 height:18
			colorpicker DLGlastUsedText03 "" pos:[259,311] width:40 height:18
			colorpicker DLGtitleText03 "" pos:[259,331] width:40 height:18
			colorpicker DLGtitleBackground03 "" pos:[259,351] width:40 height:18
			colorpicker DLGdisabledHighlight04 "" pos:[309,171] width:40 height:18
			colorpicker DLGdisabledShadow04 "" pos:[309,191] width:40 height:18
			colorpicker DLGborder04 "" pos:[309,211] width:40 height:18
			colorpicker DLGtext04 "" pos:[309,231] width:40 height:18
			colorpicker DLGbackground04 "" pos:[309,251] width:40 height:18
			colorpicker DLGhighlightedText04 "" pos:[309,271] width:40 height:18
			colorpicker DLGhighlightedBackground04 "" pos:[309,291] width:40 height:18
			colorpicker DLGlastUsedText04 "" pos:[309,311] width:40 height:18
			colorpicker DLGtitleText04 "" pos:[309,331] width:40 height:18
			colorpicker DLGtitleBackground04 "" pos:[309,351] width:40 height:18
		groupBox DLGinteractionGroup "Interaction" pos:[6,390] width:351 height:48
			checkbox DLGuseLastItem "Clicking Title Executes Last Used Item" pos:[143,409] width:206 height:17
		groupBox DLGanimationGroup "Animation" pos:[363,3] width:181 height:89
			label DLGanimationMethodLabel "Animation Method:" pos:[372,24]
			dropdownList DLGanimationMethod "" pos:[465,20] width:74 height:21 items:#("None", "Stretch", "Fade")
			spinner DLGanimationSteps "Animation Steps (frames):" type:#integer range:[0,10,0] fieldWidth:30 pos:[377,47] height:16
			spinner DLGanimationStepTime "Animation Step Time (ms):" type:#integer range:[0,999,0] fieldWidth:30 pos:[374,68] height:16
		groupBox DLGdisplayGroup "Display" pos:[551,3] width:199 height:89
			spinner DLGsubMenuTime "Sub-Menu Pause Time (ms):" type:#integer range:[0,3000,0] fieldWidth:40 pos:[558,20] height:16
			spinner DLGopacity "QuadMenu Opacity:" type:#integer range:[0,100,0] fieldWidth:40 pos:[596,40] height:16
			checkbox DLGremoveRedundantSeparators "Remove Redundant Separators" pos:[571,64] width:171 height:19
		groupBox DLGpositionGroup "Position" pos:[363,97] width:387 height:97
			bitmap DLGcursorPosBitmap "" bitmap:cursorPosBitmap pos:[382,123] width:51 height:51
			label DLGinitialCursorPosLabel "Initial Cursor Position:" pos:[444,118] width:105 height:13
			spinner DLGinitialCursorPosX "X:" range:[-10,10,0.0] scale:0.01 fieldWidth:55 pos:[466,138] height:16
			spinner DLGinitialCursorPosY "Y:" range:[-10,10,0.0] scale:0.01 fieldWidth:55 pos:[466,158] height:16
			checkbox DLGmirrorQuad "Mirror Quad" pos:[567,110] width:77 height:17
			checkbox DLGrepositionWhenClipped "Reposition Quad When Clipped" pos:[567,130] width:170 height:17
			checkbox DLGmoveCursorOnReposition "Move Cursor On Reposition" pos:[567,150] width:170 height:17
			checkbox DLGreturnCursorAfterReposition "Return Cursor After Reposition" pos:[567,170] width:170 height:17
		groupBox DLGsizeGroup "Size" pos:[363,199] width:387 height:81
			spinner DLGborderSize "Border Size (points):" type:#integer range:[1,20,1] fieldWidth:30 pos:[605,213] height:16
			checkbox DLGuniformWidth "Uniform Quad Width" pos:[375,234] width:116 height:17
			checkbox DLGuniformHeight "Uniform Item Height" pos:[375,255] width:113 height:17
			spinner DLGhorizMargin "Horizontal Margin (points):" type:#integer range:[1,20,1] fieldWidth:30 pos:[577,234] height:16
			spinner DLGvertMargin "Vertical Margin (points):" type:#integer range:[1,20,1] fieldWidth:30 pos:[589,255] height:16
		groupBox DLGfontsGroup "Fonts" pos:[363,285] width:387 height:100
			label DLGitemFontLabel "Item Font:" pos:[373,302] width:59 height:13
			editText DLGitemFont "" pos:[371,319] width:227 height:17
			label DLGitemFontListLabel "Presets:" pos:[605,319]
			dropdownList DLGitemFontList "" items:fontList pos:[650,317] width:15 height:21
			spinner DLGitemFontSize "Size:" type:#integer range:[1,20,0] fieldWidth:30 pos:[675,319] height:16
			label DLGtitleFontLabel "Title Font:" pos:[373,342] width:53 height:13
			editText DLGtitleFont "" pos:[371,359] width:227 height:17
			label DLGtitleFontListLabel "Presets:" pos:[605,359]
			dropdownList DLGtitleFontList "" items:fontList pos:[650,357] width:15 height:21
			spinner DLGtitleFontSize "Size:" type:#integer range:[1,20,0] fieldWidth:30 pos:[675,359] height:16
			checkbox DLGuseItemFont "Use Item Font" pos:[513,342] width:86 height:14
		groupBox DLGquadCombosGroup "Key Combo Assignment" pos:[363,390] width:387 height:48
			dropdownList DLGquadKeyCombo "" items:quadKeyComboStrings pos:[374,408] width:100
			label DLGquadMenuNamesLabel "Assigned To QuadMenu:" pos:[478,410]
			dropdownList DLGquadMenuNames "" items:#() pos:[602,408] width:140

		-- settings
		on DLGloadStartup pressed do (
			LoadStartupFile()
			ResetUI()
		)
		on DLGsaveStartup pressed do (
			SaveStartupFile()
			UpdateUI()
		)
--		on DLGreset pressed do (
--			needSave = true
--			QMS.ResetDefaults()
--			ResetUI()
--		)

		-- color
		on DLGimportUIcolors pressed do ( needSave = true; ImportUIColors(); UpdateUI() )
		on DLGdisabledHighlightlock changed state do ( needSave = true; UpdateQuadColor 1 DLGdisabledHighlight01; UpdateUI() )
		on DLGdisabledShadowLock changed state do ( needSave = true; UpdateQuadColor 1 DLGdisabledShadow01; UpdateUI() )
		on DLGborderLock changed state do ( needSave = true; UpdateQuadColor 1 DLGborder01; UpdateUI() )
		on DLGtextLock changed state do ( needSave = true; UpdateQuadColor 1 DLGtext01; UpdateUI() )
		on DLGbackgroundLock changed state do ( needSave = true; UpdateQuadColor 1 DLGbackground01; UpdateUI() )
		on DLGhighlightedTextLock changed state do ( needSave = true; UpdateQuadColor 1 DLGhighlightedText01; UpdateUI() )
		on DLGhighlightedBackgroundLock changed state do ( needSave = true; UpdateQuadColor 1 DLGhighlightedBackground01; UpdateUI() )
		on DLGlastUsedTextLock changed state do ( needSave = true; UpdateQuadColor 1 DLGlastUsedText01; UpdateUI() )
		on DLGtitleTextLock changed state do ( needSave = true; UpdateQuadColor 1 DLGtitleText01; UpdateUI() )
		on DLGtitleBackgroundLock changed state do ( needSave = true; UpdateQuadColor 1 DLGtitleBackground01; UpdateUI() )
		on DLGdisabledHighlight01 changed col do ( needSave = true; UpdateQuadColor 1 DLGdisabledHighlight01 )
		on DLGdisabledHighlight02 changed col do ( needSave = true; UpdateQuadColor 2 DLGdisabledHighlight02 )
		on DLGdisabledHighlight03 changed col do ( needSave = true; UpdateQuadColor 3 DLGdisabledHighlight03 )
		on DLGdisabledHighlight04 changed col do ( needSave = true; UpdateQuadColor 4 DLGdisabledHighlight04 )
		on DLGdisabledShadow01 changed col do ( needSave = true; UpdateQuadColor 1 DLGdisabledShadow01 )
		on DLGdisabledShadow02 changed col do ( needSave = true; UpdateQuadColor 2 DLGdisabledShadow02 )
		on DLGdisabledShadow03 changed col do ( needSave = true; UpdateQuadColor 3 DLGdisabledShadow03 )
		on DLGdisabledShadow04 changed col do ( needSave = true; UpdateQuadColor 4 DLGdisabledShadow04 )
		on DLGborder01 changed col do ( needSave = true; UpdateQuadColor 1 DLGborder01; UpdateBitmaps() )
		on DLGborder02 changed col do ( needSave = true; UpdateQuadColor 2 DLGborder02; UpdateBitmaps() )
		on DLGborder03 changed col do ( needSave = true; UpdateQuadColor 3 DLGborder03; UpdateBitmaps() )
		on DLGborder04 changed col do ( needSave = true; UpdateQuadColor 4 DLGborder04; UpdateBitmaps() )
		on DLGtext01 changed col do ( needSave = true; UpdateQuadColor 1 DLGtext01 )
		on DLGtext02 changed col do ( needSave = true; UpdateQuadColor 2 DLGtext02 )
		on DLGtext03 changed col do ( needSave = true; UpdateQuadColor 3 DLGtext03 )
		on DLGtext04 changed col do ( needSave = true; UpdateQuadColor 4 DLGtext04 )
		on DLGbackground01 changed col do ( needSave = true; UpdateQuadColor 1 DLGbackground01 )
		on DLGbackground02 changed col do ( needSave = true; UpdateQuadColor 2 DLGbackground02 )
		on DLGbackground03 changed col do ( needSave = true; UpdateQuadColor 3 DLGbackground03 )
		on DLGbackground04 changed col do ( needSave = true; UpdateQuadColor 4 DLGbackground04 )
		on DLGhighlightedText01 changed col do ( needSave = true; UpdateQuadColor 1 DLGhighlightedText01 )
		on DLGhighlightedText02 changed col do ( needSave = true; UpdateQuadColor 2 DLGhighlightedText02 )
		on DLGhighlightedText03 changed col do ( needSave = true; UpdateQuadColor 3 DLGhighlightedText03 )
		on DLGhighlightedText04 changed col do ( needSave = true; UpdateQuadColor 4 DLGhighlightedText04 )
		on DLGhighlightedBackground01 changed col do ( needSave = true; UpdateQuadColor 1 DLGhighlightedBackground01; UpdateBitmaps() )
		on DLGhighlightedBackground02 changed col do ( needSave = true; UpdateQuadColor 2 DLGhighlightedBackground02; UpdateBitmaps() )
		on DLGhighlightedBackground03 changed col do ( needSave = true; UpdateQuadColor 3 DLGhighlightedBackground03; UpdateBitmaps() )
		on DLGhighlightedBackground04 changed col do ( needSave = true; UpdateQuadColor 4 DLGhighlightedBackground04; UpdateBitmaps() )
		on DLGlastUsedText01 changed col do ( needSave = true; UpdateQuadColor 1 DLGlastUsedText01 )
		on DLGlastUsedText02 changed col do ( needSave = true; UpdateQuadColor 2 DLGlastUsedText02 )
		on DLGlastUsedText03 changed col do ( needSave = true; UpdateQuadColor 3 DLGlastUsedText03 )
		on DLGlastUsedText04 changed col do ( needSave = true; UpdateQuadColor 4 DLGlastUsedText04 )
		on DLGtitleText01 changed col do ( needSave = true; UpdateQuadColor 1 DLGtitleText01 )
		on DLGtitleText02 changed col do ( needSave = true; UpdateQuadColor 2 DLGtitleText02 )
		on DLGtitleText03 changed col do ( needSave = true; UpdateQuadColor 3 DLGtitleText03 )
		on DLGtitleText04 changed col do ( needSave = true; UpdateQuadColor 4 DLGtitleText04 )
		on DLGtitleBackground01 changed col do ( needSave = true; UpdateQuadColor 1 DLGtitleBackground01; UpdateBitmaps() )
		on DLGtitleBackground02 changed col do ( needSave = true; UpdateQuadColor 2 DLGtitleBackground02; UpdateBitmaps() )
		on DLGtitleBackground03 changed col do ( needSave = true; UpdateQuadColor 3 DLGtitleBackground03; UpdateBitmaps() )
		on DLGtitleBackground04 changed col do ( needSave = true; UpdateQuadColor 4 DLGtitleBackground04; UpdateBitmaps() )

		-- interaction
		on DLGuseLastItem changed state do ( needSave = true; QMS.SetUseLastExecutedItem state )

		-- animation
		on DLGanimationMethod selected idx do ( needSave = true; QMS.SetDisplayMethod (idx-1); UpdateUI() )
		on DLGanimationStepTime changed val do ( needSave = true; QMS.SetAnimatedStepTime val )
		on DLGanimationSteps changed val do ( needSave = true; QMS.SetAnimatedSteps val )

		-- display
		on DLGsubMenuTime changed val do ( needSave = true; QMS.SetSubMenuPauseTime val )
		on DLGopacity changed val do ( needSave = true; QMS.SetOpacity (val/100.) )
		on DLGremoveRedundantSeparators changed state do ( needSave = true; QMS.SetRemoveRedundantSeparators state )

		-- position
		on DLGinitialCursorPosX changed val do (
			needSave = true
			SetCursorPos [val,DLGinitialCursorPosY.value]
			UpdateCursorPosBitmap()
		)
		on DLGinitialCursorPosY changed val do (
			needSave = true
			SetCursorPos [DLGinitialCursorPosX.value,val]
			UpdateCursorPosBitmap()
		)
		on DLGmirrorQuad changed state do ( needSave = true; QMS.SetMirrorQuad state )
		on DLGrepositionWhenClipped changed state do ( needSave = true; QMS.SetRepositionWhenClipped state )
		on DLGmoveCursorOnReposition changed state do ( needSave = true; QMS.SetMoveCursorOnReposition state )
		on DLGreturnCursorAfterReposition changed state do ( needSave = true; QMS.SetReturnCursorAfterReposition state )

		-- size
		on DLGborderSize changed val do ( needSave = true; QMS.SetBorderSize val )
		on DLGuniformWidth changed state do ( needSave = true; QMS.SetUseUniformQuadWidth state )
		on DLGuniformHeight changed state do ( needSave = true; QMS.SetUseUniformItemHeight state )
		on DLGhorizMargin changed val do ( needSave = true; QMS.SetHorizontalMarginInPoints val )
		on DLGvertMargin changed val do ( needSave = true; QMS.SetVerticalMarginInPoints val )

		-- fonts
		on DLGitemFont entered str do (
			needSave = true
			QMS.SetItemFontFace str
			if DLGuseItemFont.checked then DLGtitleFont.entered str
			UpdateUI()
		)
		on DLGitemFontList selected idx do (
			needSave = true
			local fontName = StripSpaces DLGitemFontList.selected
			DLGitemFont.entered fontName
			if DLGuseItemFont.checked then DLGtitleFont.entered fontName
			UpdateUI()
		)
		on DLGitemFontSize changed val do (
			needSave = true
			QMS.SetItemFontSize val
			UpdateUI()
		)
		on DLGuseItemFont changed state do (
			needSave = true
			if state then QMS.SetTitleFontFace (QMS.GetItemFontFace())
			UpdateUI()
		)
		on DLGtitleFont entered str do (
			needSave = true
			QMS.SetTitleFontFace str
			UpdateUI()
		)
		on DLGtitleFontList selected idx do (
			needSave = true
			local fontName = StripSpaces DLGtitleFontList.selected
			DLGtitleFont.entered fontName
			UpdateUI()
		)
		on DLGtitleFontSize changed val do (
			needSave = true
			QMS.SetTitleFontSize val
			UpdateUI()
		)

		-- quadMenu key combo
		on DLGquadKeyCombo selected idx do ( UpdateUI() )
		on DLGquadMenuNames selected idx do (
			needSave = true
			local quadMenu = menuMan.FindQuadMenu DLGquadMenuNames.selected

			if quadMenu != undefined do (
				local keyCombo = quadKeyCombos[DLGquadKeyCombo.selection]

				menuMan.SetViewportRightClickMenu keyCombo quadMenu
			)
			UpdateUI()
		)

		on QuadMenuSettingsRollout open do (
			fontListNames = for str in fontList collect ((StripSpaces str) as name)

			-- collect the four arrays of color swatches
			colorSwatch = for quadNum in 0 to 3 collect (
				local sIdx = COLORSWATCHES_START + (COLORSWATCHES_COUNT * quadNum)
				local eIdx = sIdx + COLORSWATCHES_COUNT - 1
				for i in sIdx to eIdx collect R.controls[i]
			)
			-- collect the array of color locks
			colorLock = for i in COLORLOCKS_START to COLORLOCKS_END collect R.controls[i]

			ResetUI()
		)

		on QuadMenuSettingsRollout okToClose do (
			if (true == needSave) then (
				local str = "QuadMenu Settings have been modified.\n\nDo you want to save your changes\nto the startup file?"
				local res = YesNoCancelBox str title:"QuadMenu Settings"
				if res == #yes then
					SaveStartupFile()
				else if res == #cancel then
					return false
			)
			true
		)
	)


	try ( DestroyDialog QuadMenuSettingsRollout ) catch ()
	CreateDialog QuadMenuSettingsRollout width:762 height:441
)
